#!/bin/bash

# Start script to start/restart all containers
# Usage: ./start.sh

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Detect docker compose command
if docker compose version &> /dev/null; then
    DOCKER_COMPOSE="docker compose"
elif command -v docker-compose &> /dev/null; then
    DOCKER_COMPOSE="docker-compose"
else
    echo -e "${RED}Error: Docker Compose is not installed${NC}"
    exit 1
fi

echo -e "${GREEN}Starting EAP-TLS Provisioning services...${NC}"

# Check if .env file exists
if [ ! -f .env ]; then
    echo -e "${YELLOW}Warning: .env file not found. Creating from template...${NC}"
    if [ -f .env.example ]; then
        cp .env.example .env
        echo -e "${YELLOW}Please edit .env file with your configuration before continuing.${NC}"
        read -p "Press Enter to continue after editing .env file..."
    else
        echo -e "${RED}Error: .env.example not found${NC}"
        exit 1
    fi
fi

# Check if docker-compose.yml exists
if [ ! -f docker-compose.yml ]; then
    echo -e "${RED}Error: docker-compose.yml not found${NC}"
    exit 1
fi

# Load environment variables
if [ -f .env ]; then
    export $(cat .env | grep -v '^#' | xargs)
fi

# Set version if not set
export VERSION=${VERSION:-latest}

# Get host IP addresses and export as HOST_IPS (comma-separated)
# This allows the backend container to know the host's IP addresses
if command -v hostname &> /dev/null; then
    HOST_IPS=$(hostname -I | tr ' ' ',')
    export HOST_IPS
    echo -e "${GREEN}Detected host IPs: ${HOST_IPS}${NC}"
else
    echo -e "${YELLOW}Warning: hostname command not found, HOST_IPS not set${NC}"
fi

# Start services
echo -e "${GREEN}Starting containers...${NC}"
$DOCKER_COMPOSE up -d

# Wait for services to be healthy
echo -e "${GREEN}Waiting for services to start...${NC}"
sleep 10

# Wait for PostgreSQL to be ready
echo -e "${GREEN}Waiting for PostgreSQL to be ready...${NC}"
timeout=30
counter=0
while ! $DOCKER_COMPOSE exec -T postgres pg_isready -U ${POSTGRES_USER:-provisioning} > /dev/null 2>&1; do
    if [ $counter -ge $timeout ]; then
        echo -e "${RED}Error: PostgreSQL did not become ready in time${NC}"
        exit 1
    fi
    sleep 1
    counter=$((counter + 1))
done
echo -e "${GREEN}✓ PostgreSQL is ready${NC}"

# Run database migrations if backend is running
if $DOCKER_COMPOSE ps backend | grep -q "Up"; then
    echo -e "${GREEN}Running database migrations...${NC}"
    if $DOCKER_COMPOSE exec -T backend npx prisma migrate deploy > /dev/null 2>&1; then
        echo -e "${GREEN}✓ Database migrations completed${NC}"
    else
        echo -e "${YELLOW}⚠ Database migrations failed or already applied${NC}"
    fi
else
    echo -e "${YELLOW}⚠ Backend not running, skipping migrations (run manually: docker compose exec backend npx prisma migrate deploy)${NC}"
fi

# Check service status
echo -e "${GREEN}Service status:${NC}"
$DOCKER_COMPOSE ps

echo -e "${GREEN}✓ Services started successfully!${NC}"
echo -e "${GREEN}Backend: http://localhost:${BACKEND_PORT:-3000}${NC}"
echo -e "${GREEN}Frontend: http://localhost:${FRONTEND_PORT:-3001}${NC}"
echo -e "${GREEN}RADIUS Auth: ${RADIUS_AUTH_PORT:-1812}/udp${NC}"
echo -e "${GREEN}RADIUS Accounting: ${RADIUS_ACCT_PORT:-1813}/udp${NC}"

echo -e "${YELLOW}Note: Run './stop.sh' to stop services${NC}"
echo -e "${YELLOW}Note: Run '$DOCKER_COMPOSE logs -f' to view logs${NC}"

