#!/bin/bash

# Installation verification script
# Checks if all required dependencies are installed

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

echo -e "${GREEN}Checking system requirements...${NC}"

install_docker() {
    # Install Docker
    echo "Installing Docker..."
    
    # Check if Docker is already installed
    if command -v docker &> /dev/null; then
        echo "Docker is already installed. Skipping installation."
        return 0
    fi
    
    # Add Docker's official GPG key:
    apt-get update
    apt-get install -y ca-certificates curl
    install -m 0755 -d /etc/apt/keyrings
    curl -fsSL https://download.docker.com/linux/ubuntu/gpg -o /etc/apt/keyrings/docker.asc
    chmod a+r /etc/apt/keyrings/docker.asc

    # Add the repository to Apt sources:
    echo \
    "deb [arch=$(dpkg --print-architecture) signed-by=/etc/apt/keyrings/docker.asc] https://download.docker.com/linux/ubuntu \
    $(. /etc/os-release && echo "${UBUNTU_CODENAME:-$VERSION_CODENAME}") stable" | \
    tee /etc/apt/sources.list.d/docker.list > /dev/null
    apt-get update

    # Install the Docker packages
    apt-get install -y docker-ce docker-ce-cli containerd.io docker-buildx-plugin docker-compose-plugin

    # Create Docker log directory
    echo "Creating Docker log directory..."
    mkdir -p /var/log/docker
    chmod 777 /var/log/docker
    chown root:root /var/log/docker

    # Start and enable Docker service
    systemctl start docker
    systemctl enable docker

    # Add user to docker group (if not already in it and if SUDO_USER is set)
    if [ -n "$SUDO_USER" ] && [ "$SUDO_USER" != "root" ]; then
        echo "Adding user to docker group..."
        if groups $SUDO_USER | grep -q "\bdocker\b"; then
            echo "User $SUDO_USER is already in the docker group."
        else
            usermod -aG docker $SUDO_USER
            echo ""
            echo "=========================================="
            echo "IMPORTANT: Docker group membership updated!"
            echo ""
            echo "To apply the changes in your current session, you can:"
            echo "  1. Log out and log back in (recommended)"
            echo "  2. Run: newgrp docker"
            echo "  3. Run: su - $SUDO_USER"
            echo ""
            echo "Verifying group membership..."
            if su - $SUDO_USER -c "groups | grep -q docker" 2>/dev/null; then
                echo "✓ Group membership verified! Docker commands should work for $SUDO_USER"
            else
                echo "⚠ Group membership may not be active in current session"
            fi
            echo "=========================================="
            echo ""
        fi
    else
        echo "Note: Run 'usermod -aG docker <username>' to add a user to the docker group."
    fi
    
    echo "Docker installation completed successfully!"
}

# Check Docker
if command -v docker &> /dev/null; then
    DOCKER_VERSION=$(docker --version)
    echo -e "${GREEN}✓ Docker installed: ${DOCKER_VERSION}${NC}"
    
    # Check if Docker is running
    if docker info > /dev/null 2>&1; then
        echo -e "${GREEN}✓ Docker is running${NC}"
    else
        echo -e "${RED}✗ Docker is not running. Please start Docker.${NC}"
        exit 1
    fi
else
    echo -e "${RED}✗ Docker is not installed${NC}"
    # echo -e "${YELLOW}  Install Docker: https://docs.docker.com/get-docker/${NC}"
    # exit 1
    install_docker
fi

# Check Docker Compose
if docker compose version &> /dev/null; then
    COMPOSE_VERSION=$(docker compose version)
    echo -e "${GREEN}✓ Docker Compose installed: ${COMPOSE_VERSION}${NC}"
elif command -v docker-compose &> /dev/null; then
    COMPOSE_VERSION=$(docker-compose --version)
    echo -e "${GREEN}✓ Docker Compose installed: ${COMPOSE_VERSION}${NC}"
else
    echo -e "${RED}✗ Docker Compose is not installed${NC}"
    # echo -e "${YELLOW}  Install Docker Compose: https://docs.docker.com/compose/install/${NC}"
    # exit 1
    install_docker
fi

# Check if ports are available
check_port() {
    local port=$1
    local name=$2
    if lsof -Pi :${port} -sTCP:LISTEN -t >/dev/null 2>&1 || netstat -tuln 2>/dev/null | grep -q ":${port} "; then
        echo -e "${YELLOW}⚠ Port ${port} (${name}) is already in use${NC}"
        return 1
    else
        echo -e "${GREEN}✓ Port ${port} (${name}) is available${NC}"
        return 0
    fi
}

echo -e "${GREEN}Checking port availability...${NC}"
check_port 3000 "Backend" || true
check_port 3001 "Frontend" || true
check_port 1812 "RADIUS Auth" || true
check_port 1813 "RADIUS Accounting" || true
check_port 5433 "PostgreSQL" || true

# Check disk space (at least 5GB free)
AVAILABLE_SPACE=$(df -BG . | tail -1 | awk '{print $4}' | sed 's/G//')
if [ "$AVAILABLE_SPACE" -lt 5 ]; then
    echo -e "${YELLOW}⚠ Low disk space: ${AVAILABLE_SPACE}GB available (recommended: 5GB+)${NC}"
else
    echo -e "${GREEN}✓ Sufficient disk space: ${AVAILABLE_SPACE}GB available${NC}"
fi

# Check memory (at least 2GB)
TOTAL_MEM=$(free -g | awk '/^Mem:/{print $2}')
if [ "$TOTAL_MEM" -lt 2 ]; then
    echo -e "${YELLOW}⚠ Low memory: ${TOTAL_MEM}GB total (recommended: 2GB+)${NC}"
else
    echo -e "${GREEN}✓ Sufficient memory: ${TOTAL_MEM}GB total${NC}"
fi

echo -e "${GREEN}✓ All requirements verified!${NC}"
echo -e "${GREEN}You can now proceed with deployment.${NC}"

