# AIDA EAP-TLS Provisioning System - Installation Guide

A complete EAP-TLS device provisioning system for managing network device authentication and certificate distribution.

## Overview

This system provides automated provisioning of network devices with EAP-TLS certificates, including:
- Device registration and management
- Certificate Authority (CA) and server certificate management
- FreeRADIUS integration for 802.1X authentication
- Switch configuration management
- Web-based management interface

## System Requirements

- **Operating System**: Linux (Ubuntu 20.04+ recommended)
- **Docker**: 20.10+ with Docker Compose
- **Disk Space**: Minimum 5GB free
- **Memory**: Minimum 2GB RAM
- **Network**: Ports 3000, 3001, 1812, 1813, 5433 available

## Installation

### Step 1: Extract Installation Package

Extract the installation package to your desired location:

```bash
unzip aida-eap-tls-provisioning.zip
cd aida-eap-tls-provisioning
```

### Step 2: Verify System Requirements

Run the installation verification script:

```bash
chmod +x install.sh
./install.sh
```

This will check for:
- Docker installation and running status
- Docker Compose availability
- Port availability
- Disk space and memory

### Step 3: Configure Environment

Create a `.env` file with your configuration. You can start with a template:

```bash
cat > .env << 'EOF'
# Database Configuration
POSTGRES_USER=provisioning
POSTGRES_PASSWORD=your_secure_password_here
POSTGRES_DB=provisioning_db
POSTGRES_PORT=5433

# Application Version (use 'latest' or specific version like '0.25.0')
VERSION=latest

# Backend Configuration
NODE_ENV=production
PORT=3000
BACKEND_PORT=3000

# Frontend Configuration
FRONTEND_PORT=3001
REACT_APP_API_URL=http://localhost:3000/api

# RADIUS Configuration
RADIUS_SECRET=your_radius_secret_here
RADIUS_AUTH_PORT=1812
RADIUS_ACCT_PORT=1813

# Certificate Configuration
CA_VALIDITY_DAYS=3650
DEVICE_CERT_VALIDITY_DAYS=365

# Security (IMPORTANT: Change in production!)
JWT_SECRET=change-me-in-production-generate-random-string
SESSION_SECRET=change-me-in-production-generate-random-string
ENCRYPTION_KEY=your-encryption-key-here-generate-random-string
EOF
```

**Important**: 
- Change all default passwords and secrets before deploying to production!
- Generate strong random strings for `JWT_SECRET`, `SESSION_SECRET`, and `ENCRYPTION_KEY`
- Update `REACT_APP_API_URL` if accessing from a different host

### Step 4: Update Docker Images

Pull the latest Docker images from Docker Hub:

```bash
chmod +x update.sh
./update.sh
```

Or pull a specific version:

```bash
./update.sh 0.25.0
```

This will pull:
- `aidaplatform/aida-eap-tls-provisioning-backend:latest`
- `aidaplatform/aida-eap-tls-provisioning-frontend:latest`
- `aidaplatform/aida-eap-tls-provisioning-freeradius:latest`

### Step 5: Start Services

Start all services:

```bash
chmod +x start.sh
./start.sh
```

This will:
- Start PostgreSQL database
- Start backend API server
- Start frontend web interface
- Start FreeRADIUS server
- **Automatically run database migrations** (on first start)

**Note**: If migrations fail or you need to run them manually:
```bash
docker compose exec backend npx prisma migrate deploy
```

### Step 6: Access the System

- **Frontend**: http://localhost:3001 (or your server IP:3001)
- **Backend API**: http://localhost:3000 (or your server IP:3000)
- **RADIUS Auth**: UDP port 1812
- **RADIUS Accounting**: UDP port 1813

## Usage

### Initial Setup

1. **Access the Frontend**
   - Open http://localhost:3001 in your web browser
   - Log in with default credentials (if authentication is disabled) or create an admin user

2. **Configure RADIUS Server**
   - Navigate to the RADIUS page
   - Configure RADIUS server settings (IP, ports, secret)
   - Save configuration

3. **Set Up Certificates**
   - Navigate to RADIUS page → "Manage Certificates"
   - Generate or upload CA certificate
   - Generate server certificate (automatically generated when CA is created)

4. **Register Switches**
   - Navigate to Switches page
   - Add switches with SSH credentials
   - Configure RADIUS settings for each switch
   - Configure port settings (802.1X, MAB, DHCP snooping)

5. **Register Devices**
   - Navigate to Devices page
   - Add devices manually or import via CSV
   - Assign authentication groups
   - Provision devices with certificates

### Managing Services

#### Start Services
```bash
./start.sh
```

#### Stop Services
```bash
./stop.sh
```

#### View Logs
```bash
# All services
docker compose logs -f

# Specific service
docker compose logs -f backend
docker compose logs -f frontend
docker compose logs -f freeradius
docker compose logs -f postgres
```

#### Check Service Status
```bash
docker compose ps
```

#### Update to New Version
```bash
# Pull latest version
./update.sh

# Or pull specific version
./update.sh 0.26.0

# Restart services
./stop.sh
./start.sh
```

### Database Operations

#### Backup Database
```bash
docker compose exec postgres pg_dump -U provisioning provisioning_db > backup_$(date +%Y%m%d_%H%M%S).sql
```

#### Restore Database
```bash
docker compose exec -T postgres psql -U provisioning provisioning_db < backup.sql
```

#### Access Database
```bash
docker compose exec postgres psql -U provisioning provisioning_db
```

### CSV Device Import/Export

#### Export Devices
1. Navigate to Devices page
2. Click "Import/Export CSV"
3. Click "Download CSV"
4. File contains: `MAC Address,Authentication Group`

#### Import Devices
1. Navigate to Devices page
2. Click "Import/Export CSV"
3. Select CSV file (format: `MAC Address,Authentication Group`)
4. Click "Upload CSV"
5. Review results in dialog

## Configuration Files

### FreeRADIUS Configuration

FreeRADIUS configuration files are included in the Docker image by default. The system will work out of the box without any additional configuration files.

**Optional Customization**: If you need to customize FreeRADIUS configuration:

1. Create a `freeradius/` directory in the installation directory
2. Copy the configuration files you want to customize from the running container:
   ```bash
   docker compose exec freeradius cat /etc/freeradius/radiusd.conf > freeradius/radiusd.conf
   docker compose exec freeradius cat /etc/freeradius/clients.conf > freeradius/clients.conf
   # ... repeat for other files
   ```
3. Edit the files as needed
4. Uncomment the volume mounts in `docker-compose.yml` for the files you customized
5. Restart services: `./stop.sh && ./start.sh`

**Note**: The SQL configuration (`sql.conf`) uses environment variables for database connection. You may need to customize this file if your database credentials differ from the defaults.

## Production Deployment

### Security Checklist

Before deploying to production:

- [ ] Change all default passwords in `.env`
- [ ] Set strong `JWT_SECRET` and `SESSION_SECRET` (use random strings)
- [ ] Set `ENCRYPTION_KEY` for certificate encryption (use random string)
- [ ] Configure proper `REACT_APP_API_URL` for frontend
- [ ] Set up reverse proxy (nginx/traefik) for HTTPS
- [ ] Configure firewall rules
- [ ] Set up SSL/TLS certificates
- [ ] Configure backup strategy
- [ ] Set up monitoring and logging
- [ ] Review and secure FreeRADIUS configuration files

### Reverse Proxy Setup

For production, use a reverse proxy (nginx/traefik) in front of the services:

**Example nginx configuration:**

```nginx
server {
    listen 80;
    server_name your-domain.com;

    # Redirect to HTTPS
    return 301 https://$server_name$request_uri;
}

server {
    listen 443 ssl http2;
    server_name your-domain.com;

    ssl_certificate /path/to/cert.pem;
    ssl_certificate_key /path/to/key.pem;

    location / {
        proxy_pass http://localhost:3001;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }

    location /api {
        proxy_pass http://localhost:3000;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
}
```

### Firewall Configuration

Allow necessary ports:

```bash
# Allow HTTP/HTTPS
sudo ufw allow 80/tcp
sudo ufw allow 443/tcp

# Allow RADIUS
sudo ufw allow 1812/udp
sudo ufw allow 1813/udp

# Block direct access to backend/frontend (if using reverse proxy)
# sudo ufw deny 3000/tcp
# sudo ufw deny 3001/tcp
```

## Troubleshooting

### Services Won't Start

1. **Check Docker is running:**
   ```bash
   docker info
   ```

2. **Check port availability:**
   ```bash
   netstat -tuln | grep -E '3000|3001|1812|1813|5433'
   ```

3. **Check logs:**
   ```bash
   docker compose logs
   ```

### Database Connection Errors

1. **Verify PostgreSQL is healthy:**
   ```bash
   docker compose ps postgres
   ```

2. **Check DATABASE_URL in `.env`**

3. **Verify database credentials**

4. **Check database logs:**
   ```bash
   docker compose logs postgres
   ```

### Certificate Issues

1. **Check CA and server certificates are initialized:**
   - Access frontend → RADIUS page → Manage Certificates
   - Verify CA and server certificates exist

2. **Check FreeRADIUS can read certificates:**
   ```bash
   docker compose exec freeradius ls -la /etc/freeradius/certs
   docker compose exec freeradius ls -la /etc/freeradius/ca
   ```

3. **Check backend logs for certificate initialization:**
   ```bash
   docker compose logs backend | grep -i certificate
   ```

### Frontend Can't Connect to Backend

1. **Verify `REACT_APP_API_URL` in `.env`**
   - Should be `http://localhost:3000/api` for local access
   - Should be `http://your-server-ip:3000/api` for remote access

2. **Check backend is running:**
   ```bash
   docker compose ps backend
   ```

3. **Check backend logs:**
   ```bash
   docker compose logs backend
   ```

4. **Test backend directly:**
   ```bash
   curl http://localhost:3000/health
   ```

### FreeRADIUS Not Responding

1. **Check FreeRADIUS is running:**
   ```bash
   docker compose ps freeradius
   ```

2. **Check FreeRADIUS logs:**
   ```bash
   docker compose logs freeradius
   ```

3. **Test RADIUS connectivity:**
   ```bash
   # Install radtest if available
   radtest username password localhost 1812 testing123
   ```

## Architecture

```
┌─────────┐     ┌─────────┐     ┌──────────┐
│Frontend │────▶│ Backend │────▶│Database  │
│ (nginx) │     │ (Node)  │     │(Postgres)│
└─────────┘     └────┬────┘     └──────────┘
                     │
                     ▼
              ┌─────────────┐
              │ FreeRADIUS  │
              └──────┬──────┘
                     │
                     ▼
              ┌─────────────┐
              │   Switch    │
              └──────┬──────┘
                     │
                     ▼
              ┌─────────────┐
              │   Device    │
              └─────────────┘
```

## Support

For issues and questions:
- Check logs: `docker compose logs -f`
- Review configuration in `.env`
- Verify all services are healthy: `docker compose ps`
- Check service-specific logs for detailed error messages

## Version Information

To check the installed version:
```bash
docker compose exec backend cat /app/VERSION.txt
docker compose exec frontend cat /usr/share/nginx/html/VERSION.txt
```

## License

[Your License Here]
