#!/bin/bash

# Function to check if a container is healthy
wait_for_container() {
    local container=$1
    local max_attempts=30
    local attempt=1
    
    echo "Waiting for $container to be ready..."
    while [ $attempt -le $max_attempts ]; do
        if docker compose -f docker-compose.prod.yml ps $container | grep -q "Up"; then
            echo "$container is ready!"
            return 0
        fi
        echo "Attempt $attempt/$max_attempts: $container is not ready yet - waiting..."
        sleep 2
        attempt=$((attempt + 1))
    done
    echo "Error: $container failed to start after $max_attempts attempts"
    return 1
}

# Function to run a command with retries
run_with_retries() {
    local command=$1
    local max_attempts=3
    local attempt=1
    
    while [ $attempt -le $max_attempts ]; do
        echo "Attempt $attempt/$max_attempts: Running $command"
        if eval "$command"; then
            return 0
        fi
        echo "Command failed, waiting before retry..."
        sleep 5
        attempt=$((attempt + 1))
    done
    echo "Error: Command failed after $max_attempts attempts"
    return 1
}

# Wait for database to be ready
until docker compose -f docker-compose.prod.yml exec db pg_isready -U postgres; do
    echo "Database is not ready yet - waiting..."
    sleep 2
done

# Wait for backend to be ready
if ! wait_for_container "backend"; then
    echo "Error: Backend container failed to start properly"
    exit 1
fi

echo "All services are ready. Initializing..."

# Run migrations with retries
if ! run_with_retries "./migrate.sh"; then
    echo "Error: Database migration failed after retries"
    exit 1
fi

# Initialize admin with retries
if ! run_with_retries "./init_admin.sh"; then
    echo "Error: Admin initialization failed after retries"
    exit 1
fi

echo "Database initialized successfully."