#!/bin/bash

# Check if script is run as root
if [ "$EUID" -ne 0 ]; then
    echo "Please run this script with sudo:"
    echo "sudo ./install.sh"
    exit 1
fi

# Get APP_VERSION from command line argument, default to "latest"
APP_VERSION=${1:-"latest"}

# Set up logging with version info
LOG_FILE="/var/log/inspextor_architect_install.log"
exec > >(tee -a "$LOG_FILE") 2>&1

INSTALLER_VERSION=$(cat VERSION 2>/dev/null || echo "unknown")

echo "=== Inspextor Architect Installation v${INSTALLER_VERSION} Started $(date) ==="
echo "Installer version: ${INSTALLER_VERSION}"
echo "System: $(uname -a)"
echo "Distribution: $(lsb_release -d 2>/dev/null | cut -f2 || echo 'Unknown')"

# Check system requirements
echo "Checking system requirements..."
if [ "$(uname -m)" != "x86_64" ]; then
    echo "ERROR: This installation requires x86_64 architecture"
    exit 1
fi

# Check available disk space (at least 2GB)
# AVAILABLE_SPACE=$(df /opt | awk 'NR==2 {print $4}')
# if [ "$AVAILABLE_SPACE" -lt 2097152 ]; then
#     echo "ERROR: Insufficient disk space. Need at least 2GB available in /opt"
#     exit 1
# fi

# Determine current working directory and install to /opt
echo "Installing application to /opt/inspextor_architect..."
CURRENT_DIR=$(pwd)
echo "Current directory: $CURRENT_DIR"

# Create /opt/inspextor_architect directory if it doesn't exist
if [ ! -d "/opt/inspextor_architect" ]; then
    echo "Creating /opt/inspextor_architect directory..."
    mkdir -p /opt/inspextor_architect
else
    echo "/opt/inspextor_architect directory already exists, replacing contents..."
    # Remove existing contents
    rm -rf /opt/inspextor_architect/*
fi

# Copy the current directory contents to /opt
echo "Copying application files from $CURRENT_DIR to /opt/inspextor_architect..."
if ! cp -r "$CURRENT_DIR"/* /opt/inspextor_architect/; then
    echo "ERROR: Failed to copy application files"
    exit 1
fi

# Set proper permissions
echo "Setting proper permissions..."
chown -R root:root /opt/inspextor_architect
chmod -R 755 /opt/inspextor_architect
chmod +x /opt/inspextor_architect/*.sh

# Change to the installation directory
cd /opt/inspextor_architect

# Install Docker
echo "Installing Docker..."
# Add Docker's official GPG key:
apt-get update
apt-get install -y ca-certificates curl
install -m 0755 -d /etc/apt/keyrings
curl -fsSL https://download.docker.com/linux/ubuntu/gpg -o /etc/apt/keyrings/docker.asc
chmod a+r /etc/apt/keyrings/docker.asc

# Add the repository to Apt sources:
echo \
  "deb [arch=$(dpkg --print-architecture) signed-by=/etc/apt/keyrings/docker.asc] https://download.docker.com/linux/ubuntu \
  $(. /etc/os-release && echo "${UBUNTU_CODENAME:-$VERSION_CODENAME}") stable" | \
  tee /etc/apt/sources.list.d/docker.list > /dev/null
apt-get update

# Install the Docker packages
apt-get install -y docker-ce docker-ce-cli containerd.io docker-buildx-plugin docker-compose-plugin

# Create Docker log directory
echo "Creating Docker log directory..."
mkdir -p /var/log/docker
chmod 777 /var/log/docker
chown root:root /var/log/docker

# Add user to docker group
echo "Adding user to docker group..."
usermod -aG docker $SUDO_USER

# Apply Docker group changes
echo "Applying Docker group changes..."
newgrp docker << EONG
echo "Docker group changes applied successfully!"
EONG

# Install PostgreSQL
echo "Installing PostgreSQL..."
apt install -y postgresql postgresql-contrib

# Start PostgreSQL service
echo "Starting PostgreSQL service..."
systemctl start postgresql
systemctl enable postgresql

# Set PostgreSQL password
echo "Setting PostgreSQL password..."
sudo -u postgres psql -c "ALTER USER postgres WITH PASSWORD 'postgres';"

# Disable apache2
echo "Disabling apache2..."
systemctl disable apache2
systemctl stop apache2

# Install Python and pip
echo "Installing Python and pip..."
apt install -y python3 python3-pip

# Install Node.js
echo "Installing Node.js..."
curl -fsSL https://deb.nodesource.com/setup_18.x | bash -
apt-get install -y nodejs

# Install networking utilities
echo "Installing networking utilities..."
apt-get install -y net-tools iputils-ping

# Install cron for crontab functionality
echo "Installing cron..."
apt-get install -y cron

# Create TFTP directory
echo "Creating TFTP directory..."
sudo mkdir -p /tftpboot
sudo chmod 777 /tftpboot
sudo chown root:root /tftpboot

# Install TFTP
echo "Installing TFTP Client..."
apt-get install -y tftp-hpa

# Install Avahi for mDNS support
echo "Installing Avahi for mDNS support..."
apt-get install -y avahi-daemon avahi-utils

# Get the server's IP address
# SERVER_IP=$(hostname -I | awk '{print $1}')

# Configure Avahi hosts file for architect.local resolution
# echo "Configuring Avahi hosts for architect.local resolution..."

# Create file if it doesn't exist
# touch /etc/avahi/hosts

# Remove any existing architect.local entry
# sed -i '/architect\.local/d' /etc/avahi/hosts

# Add the new architect.local entry
# echo "$SERVER_IP architect.local" >> /etc/avahi/hosts

# Configure Avahi to publish architect.local
echo "Configuring Avahi for architect.local..."
cat > /etc/avahi/services/architect.service << 'EOF'
<?xml version="1.0" standalone='no'?>
<!DOCTYPE service-group SYSTEM "avahi-service.dtd">
<service-group>
  <name replace-wildcards="yes">architect</name>
  <service>
    <type>_http._tcp</type>
    <port>80</port>
  </service>
  <service>
    <type>_https._tcp</type>
    <port>443</port>
  </service>
  <service>
    <type>_mqtt._tcp</type>
    <port>1883</port>
  </service>
  <service>
    <type>_mqtt._tcp</type>
    <port>8883</port>
  </service>
  <service>
    <type>_mqtt._tcp</type>
    <port>9001</port>
  </service>
</service-group>
EOF

# Start and enable Avahi
systemctl start avahi-daemon
systemctl enable avahi-daemon

# Install services
#echo "Installing services..."
#./install_services.sh

# Ensure we're in the correct directory for running scripts
cd /opt/inspextor_architect

# Verify required scripts exist
echo "Verifying required scripts..."
for script in update.sh init_db.sh stop.sh run.sh host_controller_start.sh pull.sh; do
    if [ ! -f "$script" ]; then
        echo "ERROR: Required script $script not found in /opt/inspextor_architect"
        exit 1
    fi
    echo "✓ Found $script"
done

# Make sure all scripts are executable
echo "Making scripts executable..."
chmod +x *.sh

# Update the server - pull and run latest images
echo "Updating server with version $APP_VERSION..."
./pull.sh $APP_VERSION

# Reset the Database
echo "Resetting database..."
./reset_database.sh

# Start the database
echo "Starting database..."
docker compose -f docker-compose.prod.yml up -d db

# Wait for database to be ready before running migrations
echo "Waiting for database to be ready..."
until docker compose -f docker-compose.prod.yml exec db pg_isready -U postgres; do
    echo "Database is not ready yet - waiting..."
    sleep 2
done

# Run migrations directly using alembic
echo "Running migrations..."
docker compose -f docker-compose.prod.yml run --rm backend alembic upgrade head

# Setup Crontab
echo "Setting up Crontab..."

# Ensure cron service is running
echo "Starting cron service..."
systemctl start cron
systemctl enable cron

# Remove any existing crontab entries for this application
(crontab -l 2>/dev/null | grep -v "inspextor_architect") | crontab -

# Add new crontab entries
(crontab -l 2>/dev/null; echo "@reboot cd /opt/inspextor_architect/ && /opt/inspextor_architect/run.sh >> /var/log/run_script_startup.log 2>&1") | crontab -
(crontab -l 2>/dev/null; echo "@reboot cd /opt/inspextor_architect/ && /opt/inspextor_architect/host_controller_start.sh >> /var/log/host_controller_startup.log 2>&1") | crontab -
(crontab -l 2>/dev/null; echo "@reboot cd /opt/inspextor_architect/ && /opt/inspextor_architect/avahi-publish-architect.sh >> /var/log/avahi_publish_startup.log 2>&1") | crontab -

# Create application log directory
echo "Creating application log directory..."
mkdir -p /var/log/inspextor_architect
chmod 755 /var/log/inspextor_architect

# Create a systemd service file for better service management
echo "Creating systemd service file..."
cat > /etc/systemd/system/inspextor-architect.service << EOF
[Unit]
Description=Inspextor Architect Application (v${INSTALLER_VERSION})
After=docker.service postgresql.service
Requires=docker.service postgresql.service

[Service]
Type=oneshot
RemainAfterExit=yes
WorkingDirectory=/opt/inspextor_architect
ExecStart=/opt/inspextor_architect/run.sh
ExecStop=/opt/inspextor_architect/stop.sh
User=root
Group=root
Environment=INSTALLER_VERSION=${INSTALLER_VERSION}

[Install]
WantedBy=multi-user.target
EOF

# Enable the service
systemctl daemon-reload
systemctl enable inspextor-architect.service

# Generate self-signed certificates
echo "Generating self-signed certificates..."

# Generate CA certificate
openssl genrsa -out mosquitto/certs/ca.key 2048
openssl req -new -x509 -days 365 -key mosquitto/certs/ca.key -out mosquitto/certs/ca.crt -subj "/C=US/ST=State/L=City/O=MHT-Technologies/CN=MQTT-CA"

# Generate server certificate for architect.local
openssl genrsa -out mosquitto/certs/server.key 2048
openssl req -new -key mosquitto/certs/server.key -out mosquitto/certs/server.csr -subj "/C=US/ST=State/L=City/O=MHT-Technologies/CN=architect.local"

# Sign server certificate
openssl x509 -req -in mosquitto/certs/server.csr -CA mosquitto/certs/ca.crt -CAkey mosquitto/certs/ca.key -CAcreateserial -out mosquitto/certs/server.crt -days 365

# Set proper permissions
chmod 600 mosquitto/certs/*.key
chmod 644 mosquitto/certs/*.crt

# Create Mosquitto password file using the container
echo "Creating Mosquitto password file..."
printf "admin\nadmin\n" | docker run --rm -i -v $(pwd)/mosquitto/config:/mosquitto/config eclipse-mosquitto:2.0 mosquitto_passwd -c /mosquitto/config/password_file admin

# Fix password file permissions AFTER creation
echo "Fixing password file permissions..."
chmod 700 mosquitto/config/password_file
chown root:root mosquitto/config/password_file

# Start the server
echo "Starting server..."
./run.sh

# Initialize the Admin User
echo "Initializing admin user..."
./init_admin.sh

# Stop the server because it is running as root
# echo "Stopping server..."
# if ! ./stop.sh; then
#     echo "WARNING: Failed to stop server (this might be normal if server wasn't running)"
# fi

echo "=== Installation Summary ==="
echo "✓ Application installed to: /opt/inspextor_architect"
echo "✓ Log files location: /var/log/inspextor_architect"
echo "✓ Installation log: $LOG_FILE"
echo "✓ Systemd service: inspextor-architect.service"
echo "✓ Crontab entries configured for auto-startup"
echo ""
echo "Installation complete! Rebooting server to apply all changes and start the application automatically..."
echo "The server will be available after reboot at the configured address."
echo "To check service status after reboot: systemctl status inspextor-architect"
echo "To view logs: journalctl -u inspextor-architect -f"
echo ""
echo "Rebooting in 3 seconds... Press Ctrl+C to cancel."
sleep 3
reboot


