#!/usr/bin/env python3
from flask import Flask, request, jsonify, send_file
import subprocess
import threading
import logging
import os

path = os.path.dirname(os.path.abspath(__file__))

# Set up logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

app = Flask(__name__)

@app.route('/systemctl/<action>/<service>', methods=['POST'])
def systemctl_action(action, service):
    try:
        logger.info(f"Received request: systemctl {action} {service}")
        result = subprocess.run(
            ['systemctl', action, service],
            capture_output=True,
            text=True,
            timeout=30
        )
        logger.info(f"systemctl {action} {service} completed with return code {result.returncode}")
        return jsonify({
            'success': result.returncode == 0,
            'stdout': result.stdout,
            'stderr': result.stderr,
            'returncode': result.returncode
        })
    except Exception as e:
        logger.error(f"Error executing systemctl {action} {service}: {str(e)}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/restart', methods=['POST'])
def restart_containers():
    try:
        logger.info("Received restart request")
        
        # Change to the correct directory and run restart script
        script_dir = path
        script_path = os.path.join(script_dir, 'run.sh')
        
        result = subprocess.run(
            ['bash', script_path],
            capture_output=True,
            text=True,
            timeout=120,
            cwd=script_dir            
        )
        
        logger.info(f"Restart script completed with return code {result.returncode}")
        return jsonify({
            'success': result.returncode == 0,
            'stdout': result.stdout,
            'stderr': result.stderr,
            'returncode': result.returncode
        })
    except Exception as e:
        logger.error(f"Error executing restart script: {str(e)}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/update', methods=['POST'])
def update_system():
    try:
        logger.info("Received update request")
        
        # Parse JSON body for APP_VERSION
        data = request.get_json() or {}
        app_version = data.get('APP_VERSION', 'latest')
        
        logger.info(f"Updating to version: {app_version}")
        
        # Change to the correct directory and run update script
        script_dir = path
        script_path = os.path.join(script_dir, 'update.sh')

        result = subprocess.run(
            ['bash', script_path, app_version],
            capture_output=True,
            text=True,
            timeout=300,  # Longer timeout for updates
            cwd=script_dir            
        )
        
        logger.info(f"Update script completed with return code {result.returncode}")
        return jsonify({
            'success': result.returncode == 0,
            'stdout': result.stdout,
            'stderr': result.stderr,
            'returncode': result.returncode
        })
    except Exception as e:
        logger.error(f"Error executing update script: {str(e)}")
        return jsonify({'success': False, 'error': str(e)}), 500
    
@app.route('/health', methods=['GET'])
def health():
    return jsonify({'status': 'healthy'})

@app.route('/certificates/ca.crt', methods=['GET'])
def get_ca_certificate():
    """Serve the CA certificate for MQTT clients"""
    try:
        ca_cert_path = os.path.join(path, 'mosquitto', 'certs', 'ca.crt')
        
        if not os.path.exists(ca_cert_path):
            logger.error(f"CA certificate not found at {ca_cert_path}")
            return jsonify({'error': 'CA certificate not found'}), 404
        
        logger.info(f"Serving CA certificate from {ca_cert_path}")
        return send_file(
            ca_cert_path,
            mimetype='application/x-pem-file',
            as_attachment=True,
            download_name='ca.crt'
        )
    except Exception as e:
        logger.error(f"Error serving CA certificate: {str(e)}")
        return jsonify({'error': str(e)}), 500

@app.route('/certificates/info', methods=['GET'])
def get_certificate_info():
    """Get certificate information"""
    try:
        ca_cert_path = os.path.join(path, 'mosquitto', 'certs', 'ca.crt')
        server_cert_path = os.path.join(path, 'mosquitto', 'certs', 'server.crt')
        
        info = {
            'ca_certificate_available': os.path.exists(ca_cert_path),
            'server_certificate_available': os.path.exists(server_cert_path),
            'mqtt_broker': 'architect.local',
            'mqtt_port': 8883,
            'mqtt_websocket_port': 9001
        }
        
        if os.path.exists(ca_cert_path):
            # Get certificate expiration info
            result = subprocess.run(
                ['openssl', 'x509', '-in', ca_cert_path, '-noout', '-dates'],
                capture_output=True,
                text=True
            )
            if result.returncode == 0:
                info['ca_certificate_dates'] = result.stdout.strip()
        
        return jsonify(info)
    except Exception as e:
        logger.error(f"Error getting certificate info: {str(e)}")
        return jsonify({'error': str(e)}), 500

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=5002)