# Inspextor Architect - Installation & Management Guide

## Overview

Inspextor Architect is a comprehensive IoT device management platform that includes:
- Web-based frontend (React)
- Backend API (Flask/Python)
- PostgreSQL database
- MQTT broker (Eclipse Mosquitto)
- TFTP server
- Automatic SSL/TLS certificate management

## Quick Start

### Prerequisites
- Ubuntu 22.04 (x86_64)
- Root/sudo access
- Internet connection for initial setup
- At least 2GB available disk space

### Installation

```bash
# Clone or download the installation package
cd /path/to/inspextor_architect

# Run the installer
sudo ./install.sh [VERSION]
```

**Note:** Replace `[VERSION]` with the desired version (e.g., `latest`, `v1.2.3`)

## Management Commands

All commands should be run from `/opt/inspextor_architect/` directory.

### Start Services
```bash
cd /opt/inspextor_architect
./run.sh
```

### Stop Services
```bash
cd /opt/inspextor_architect
./stop.sh
```

### Restart Services
```bash
cd /opt/inspextor_architect
./stop.sh
./run.sh
```

### Update Application
```bash
cd /opt/inspextor_architect
./update.sh [VERSION]
```

### Pull Latest Images
```bash
cd /opt/inspextor_architect
./pull.sh [VERSION]
```

### Uninstall
```bash
cd /opt/inspextor_architect
./uninstall.sh
```

## Service Management

### Systemd Service
The application is installed as a systemd service for automatic startup.

```bash
# Check service status
systemctl status inspextor-architect

# Start service manually
systemctl start inspextor-architect

# Stop service manually
systemctl stop inspextor-architect

# Enable auto-start on boot
systemctl enable inspextor-architect

# Disable auto-start on boot
systemctl disable inspextor-architect

# View service logs
journalctl -u inspextor-architect -f
```

### Docker Compose
Direct Docker Compose commands:

```bash
cd /opt/inspextor_architect

# Start all services
docker compose -f docker-compose.prod.yml up -d

# Stop all services
docker compose -f docker-compose.prod.yml down

# View logs
docker compose -f docker-compose.prod.yml logs -f

# Restart specific service
docker compose -f docker-compose.prod.yml restart frontend
docker compose -f docker-compose.prod.yml restart backend
docker compose -f docker-compose.prod.yml restart mqtt
```

## Network Discovery (mDNS)

### mDNS Support
The application uses Avahi-daemon to provide mDNS support, allowing devices to discover the service using `architect.local`.

### Access Methods
- **mDNS**: `https://architect.local` (recommended)
- **Localhost**: `https://localhost`
- **Direct IP**: `https://[server-ip]`

### Testing mDNS Resolution
```bash
# Test mDNS resolution
avahi-resolve -n architect.local

# Browse available services
avahi-browse -at

# Test from another device on the network
ping architect.local
```

### Troubleshooting mDNS
```bash
# Check Avahi status
systemctl status avahi-daemon

# View Avahi logs
journalctl -u avahi-daemon -f

# Restart Avahi
systemctl restart avahi-daemon

# Check published services
avahi-browse -at | grep architect
```
```

### 3. Alternative: Use hostname instead of architect.local

If you prefer to use the system's hostname instead of `architect.local`, you can:

```bash
# Get the system hostname
HOSTNAME=$(hostname)

# Update certificate generation to use hostname
openssl req -new -key mosquitto/certs/server.key -out mosquitto/certs/server.csr -subj "/C=US/ST=State/L=City/O=MHT-Technologies/CN=$HOSTNAME"
```

### 4. Update certificate generation in install.sh

Modify the certificate generation section:

```bash:architect_scripts/inspextor_architect/install.sh
# Generate self-signed certificates
echo "Generating self-signed certificates..."

# Create certificate directories
mkdir -p mosquitto/certs mosquitto/config mosquitto/data mosquitto/logs

# Get hostname for certificate
HOSTNAME=$(hostname)

# Generate CA certificate
openssl genrsa -out mosquitto/certs/ca.key 2048
openssl req -new -x509 -days 365 -key mosquitto/certs/ca.key -out mosquitto/certs/ca.crt -subj "/C=US/ST=State/L=City/O=MHT-Technologies/CN=MQTT-CA"

# Generate server certificate for architect.local and hostname
openssl genrsa -out mosquitto/certs/server.key 2048
openssl req -new -key mosquitto/certs/server.key -out mosquitto/certs/server.csr -subj "/C=US/ST=State/L=City/O=MHT-Technologies/CN=architect.local"

# Create config file for multiple domains
cat > mosquitto/certs/server.conf << EOF
[req]
distinguished_name = req_distinguished_name
req_extensions = v3_req
prompt = no

[req_distinguished_name]
C = US
ST = State
L = City
O = MHT-Technologies
CN = architect.local

[v3_req]
keyUsage = keyEncipherment, dataEncipherment
extendedKeyUsage = serverAuth
subjectAltName = @alt_names

[alt_names]
DNS.1 = architect.local
DNS.2 = $HOSTNAME
DNS.3 = localhost
IP.1 = 127.0.0.1
EOF

# Sign server certificate with multiple domains
openssl x509 -req -in mosquitto/certs/server.csr -CA mosquitto/certs/ca.crt -CAkey mosquitto/certs/ca.key -CAcreateserial -out mosquitto/certs/server.crt -days 365 -extfile mosquitto/certs/server.conf -extensions v3_req

# Set proper permissions
chmod 600 mosquitto/certs/*.key
chmod 644 mosquitto/certs/*.crt
```

## Benefits of this approach:

1. **Automatic discovery** - devices can find your service via `architect.local`
2. **Multiple access methods** - works with hostname, localhost, and IP
3. **Standard mDNS** - compatible with all modern operating systems
4. **No DNS configuration** - works out of the box on local networks

This setup will make your application easily discoverable on the local network while maintaining security with SSL/TLS certificates.

## MQTT Management

### MQTT Configuration
- **Broker**: `architect.local` or `localhost`
- **Port**: `8883` (MQTTS) / `9001` (WebSocket MQTTS) / `1883` (MQTT)
- **Authentication**: Anonymous access enabled
- **SSL/TLS**: Self-signed certificates

### Create/Update MQTT Users

#### Add New User
```bash
cd /opt/inspextor_architect

# Add user (will prompt for password)
mosquitto_passwd mosquitto/config/password_file username

# Example: Add user 'device1'
mosquitto_passwd mosquitto/config/password_file device1
```

#### Remove User
```bash
cd /opt/inspextor_architect

# Remove user
mosquitto_passwd -D mosquitto/config/password_file username

# Example: Remove user 'device1'
mosquitto_passwd -D mosquitto/config/password_file device1
```

#### Change User Password
```bash
cd /opt/inspextor_architect

# Change password (will prompt for new password)
mosquitto_passwd mosquitto/config/password_file username

# Example: Change password for 'device1'
mosquitto_passwd mosquitto/config/password_file device1
```

#### List All Users
```bash
cd /opt/inspextor_architect

# View password file (shows usernames only)
cat mosquitto/config/password_file
```

### MQTT Testing

#### Test Connection (Anonymous)
```bash
# Publish message
mosquitto_pub -h architect.local -p 8883 -t "test/topic" -m "Hello MQTT" --cafile mosquitto/certs/ca.crt

# Subscribe to topic
mosquitto_sub -h architect.local -p 8883 -t "test/topic" --cafile mosquitto/certs/ca.crt
```

#### Test Connection (Authenticated)
```bash
# Publish message with credentials
mosquitto_pub -h architect.local -p 8883 -t "test/topic" -m "Hello MQTT" --cafile mosquitto/certs/ca.crt -u username -P password

# Subscribe with credentials
mosquitto_sub -h architect.local -p 8883 -t "test/topic" --cafile mosquitto/certs/ca.crt -u username -P password
```

## SSL/TLS Certificates

### Certificate Information
- **Type**: Self-signed certificates
- **Domain**: `architect.local`
- **Validity**: 365 days
- **Location**: `/opt/inspextor_architect/mosquitto/certs/`

### Regenerate Certificates
```bash
cd /opt/inspextor_architect

# Backup existing certificates
cp -r mosquitto/certs mosquitto/certs.backup.$(date +%Y%m%d)

# Generate new certificates
openssl genrsa -out mosquitto/certs/ca.key 2048
openssl req -new -x509 -days 365 -key mosquitto/certs/ca.key -out mosquitto/certs/ca.crt -subj "/C=US/ST=State/L=City/O=MHT-Technologies/CN=MQTT-CA"

openssl genrsa -out mosquitto/certs/server.key 2048
openssl req -new -key mosquitto/certs/server.key -out mosquitto/certs/server.csr -subj "/C=US/ST=State/L=City/O=MHT-Technologies/CN=architect.local"

openssl x509 -req -in mosquitto/certs/server.csr -CA mosquitto/certs/ca.crt -CAkey mosquitto/certs/ca.key -CAcreateserial -out mosquitto/certs/server.crt -days 365

# Set permissions
chmod 600 mosquitto/certs/*.key
chmod 644 mosquitto/certs/*.crt

# Restart services to apply new certificates
./restart.sh
```

## Database Management

### Reset Database
```bash
cd /opt/inspextor_architect
./reset_database.sh
```

### Initialize Admin User
```bash
cd /opt/inspextor_architect
./init_admin.sh
```

### Database Migrations
```bash
cd /opt/inspextor_architect

# Run migrations
docker compose -f docker-compose.prod.yml run --rm backend alembic upgrade head

# Check migration status
docker compose -f docker-compose.prod.yml run --rm backend alembic current
```

## Logs and Monitoring

### Application Logs
```bash
# View installation log
tail -f /var/log/inspextor_architect_install.log

# View startup logs
tail -f /var/log/run_script_startup.log
tail -f /var/log/host_controller_startup.log

# View systemd service logs
journalctl -u inspextor-architect -f
```

### Docker Logs
```bash
cd /opt/inspextor_architect

# View all service logs
docker compose -f docker-compose.prod.yml logs -f

# View specific service logs
docker compose -f docker-compose.prod.yml logs -f frontend
docker compose -f docker-compose.prod.yml logs -f backend
docker compose -f docker-compose.prod.yml logs -f mqtt
docker compose -f docker-compose.prod.yml logs -f db
```

### MQTT Logs
```bash
# View MQTT broker logs
tail -f /opt/inspextor_architect/mosquitto/logs/mosquitto.log
```

## Network Access

### Default Ports
- **Frontend (HTTP)**: `80`
- **Frontend (HTTPS)**: `443`
- **Backend API**: `5001`
- **Database**: `5434`
- **MQTT (MQTTS)**: `8883`
- **MQTT (WebSocket)**: `9001`
- **TFTP**: `69`

### Access URLs
- **Web Interface**: `https://architect.local` or `https://localhost`
- **API Documentation**: `https://architect.local/swaggerui/`
- **Backend API**: `https://architect.local/api/`

## Troubleshooting

### Common Issues

#### Services Not Starting
```bash
# Check Docker status
systemctl status docker

# Check available disk space
df -h

# Check Docker logs
docker compose -f docker-compose.prod.yml logs
```

#### MQTT Connection Issues
```bash
# Check MQTT service status
docker compose -f docker-compose.prod.yml ps mqtt

# Check MQTT logs
docker compose -f docker-compose.prod.yml logs mqtt

# Test MQTT connectivity
mosquitto_pub -h localhost -p 8883 -t "test" -m "test" --cafile mosquitto/certs/ca.crt
```

#### Certificate Issues
```bash
# Check certificate validity
openssl x509 -in mosquitto/certs/server.crt -text -noout

# Check certificate expiration
openssl x509 -in mosquitto/certs/server.crt -noout -dates
```

### Reset Everything
```bash
cd /opt/inspextor_architect

# Stop all services
./stop.sh

# Reset database
./reset_database.sh

# Regenerate certificates
# (see SSL/TLS Certificates section above)

# Start services
./run.sh

# Initialize admin user
./init_admin.sh
```

## Support

For issues and support:
- Check logs in `/var/log/inspextor_architect/`
- Review Docker logs: `docker compose -f docker-compose.prod.yml logs`
- Contact MHT Technologies support team

## Version History

- See [CHANGELOG.md](CHANGELOG.md) for detailed version history
